<?php
// models/Venta.php

class Venta {
    private $db;
    private $pdo;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->pdo = $this->db->getConnection();
    }

    /**
     * Crea una nueva venta y sus detalles en una transacción.
     *
     * @param int $idCliente
     * @param float $total
     * @param array $productos
     * @return int|false El ID de la venta creada o false si falla.
     */
    public function create($idCliente, $total, $productos) {
        try {
            $this->pdo->beginTransaction();

            // 1. Insertar la venta en la tabla 'ventas'
            $stmtVenta = $this->pdo->prepare(
                "INSERT INTO ventas (id_cliente, total) VALUES (:id_cliente, :total)"
            );
            $stmtVenta->bindParam(':id_cliente', $idCliente, PDO::PARAM_INT);
            $stmtVenta->bindParam(':total', $total);
            $stmtVenta->execute();
            
            $idVenta = $this->pdo->lastInsertId();

            // 2. Insertar cada producto en 'detalle_ventas' y actualizar el stock en 'productos'
            $stmtDetalle = $this->pdo->prepare(
                "INSERT INTO detalle_ventas (id_venta, id_producto, cantidad, precio_unitario) VALUES (:id_venta, :id_producto, :cantidad, :precio_unitario)"
            );
            
            $stmtUpdateStock = $this->pdo->prepare(
                "UPDATE productos SET stock = stock - :cantidad WHERE id = :id_producto"
            );

            foreach ($productos as $producto) {
                // Insertar detalle
                $stmtDetalle->bindParam(':id_venta', $idVenta, PDO::PARAM_INT);
                $stmtDetalle->bindParam(':id_producto', $producto['id'], PDO::PARAM_INT);
                $stmtDetalle->bindParam(':cantidad', $producto['cantidad'], PDO::PARAM_STR);
                $stmtDetalle->bindParam(':precio_unitario', $producto['precio']);
                $stmtDetalle->execute();

                // Actualizar stock
                $stmtUpdateStock->bindParam(':cantidad', $producto['cantidad'], PDO::PARAM_STR);
                $stmtUpdateStock->bindParam(':id_producto', $producto['id'], PDO::PARAM_INT);
                $stmtUpdateStock->execute();
            }

            // 3. Si todo fue bien, confirmar la transacción
            $this->pdo->commit();
            
            return $idVenta;

        } catch (Exception $e) {
            // 4. Si algo falla, revertir la transacción
            $this->pdo->rollBack();
            // Log del error (en un sistema real)
            // error_log($e->getMessage());
            return false;
        }
    }
    
    public function getVentaById($id) {
        $query = "
            SELECT 
                v.id, v.fecha_venta, v.total,
                c.nombre as cliente_nombre, c.direccion as cliente_direccion, c.telefono as cliente_telefono, c.email as cliente_email
            FROM ventas v
            JOIN clientes c ON v.id_cliente = c.id
            WHERE v.id = :id
        ";
        $stmt = $this->pdo->prepare($query);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getDetallesByVentaId($idVenta) {
        $query = "
            SELECT 
                d.cantidad, d.precio_unitario,
                p.nombre as producto_nombre, p.codigo as producto_codigo
            FROM detalle_ventas d
            JOIN productos p ON d.id_producto = p.id
            WHERE d.id_venta = :id_venta
        ";
        $stmt = $this->pdo->prepare($query);
        $stmt->bindParam(':id_venta', $idVenta, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getVentasPorFechas($fecha_inicio, $fecha_fin) {
        $query = "
            SELECT v.id, v.fecha_venta, v.total, c.nombre as cliente_nombre
            FROM ventas v
            JOIN clientes c ON v.id_cliente = c.id
            WHERE v.fecha_venta BETWEEN :fecha_inicio AND :fecha_fin
            ORDER BY v.fecha_venta DESC
        ";
        $stmt = $this->pdo->prepare($query);
        $stmt->bindParam(':fecha_inicio', $fecha_inicio);
        $stmt->bindParam(':fecha_fin', $fecha_fin);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // --- Métodos para el Dashboard ---

    public function getEstadisticasGenerales() {
        $stats = [];
        // Ventas de Hoy
        $stmtHoy = $this->pdo->prepare("SELECT SUM(total) as total FROM ventas WHERE DATE(fecha_venta) = CURDATE()");
        $stmtHoy->execute();
        $stats['hoy'] = $stmtHoy->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

        // Ventas de Este Mes
        $stmtMes = $this->pdo->prepare("SELECT SUM(total) as total FROM ventas WHERE YEAR(fecha_venta) = YEAR(CURDATE()) AND MONTH(fecha_venta) = MONTH(CURDATE())");
        $stmtMes->execute();
        $stats['mes'] = $stmtMes->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

        // Ventas de Este Año
        $stmtAnio = $this->pdo->prepare("SELECT SUM(total) as total FROM ventas WHERE YEAR(fecha_venta) = YEAR(CURDATE())");
        $stmtAnio->execute();
        $stats['anio'] = $stmtAnio->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
        
        // Total histórico
        $stmtTotal = $this->pdo->prepare("SELECT SUM(total) as total FROM ventas");
        $stmtTotal->execute();
        $stats['total_historico'] = $stmtTotal->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

        return $stats;
    }

    public function getVentasUltimosDias($dias = 7) {
        $query = "
            SELECT DATE(fecha_venta) as dia, SUM(total) as total_dia
            FROM ventas
            WHERE fecha_venta >= CURDATE() - INTERVAL :dias DAY
            GROUP BY DATE(fecha_venta)
            ORDER BY DATE(fecha_venta) ASC
        ";
        $stmt = $this->pdo->prepare($query);
        $stmt->bindParam(':dias', $dias, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getVentasMensuales($anio) {
        $query = "
            SELECT MONTH(fecha_venta) as mes, SUM(total) as total_mes
            FROM ventas
            WHERE YEAR(fecha_venta) = :anio
            GROUP BY MONTH(fecha_venta)
            ORDER BY mes ASC
        ";
        $stmt = $this->pdo->prepare($query);
        $stmt->bindParam(':anio', $anio, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getVentasAnuales() {
        $query = "
            SELECT YEAR(fecha_venta) as anio, SUM(total) as total_anio
            FROM ventas
            GROUP BY YEAR(fecha_venta)
            ORDER BY anio ASC
        ";
        $stmt = $this->pdo->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
